<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

include_once __DIR__ . "/config.php";
include_once __DIR__ . "/connection.php";

// Get input JSON
$input = json_decode(file_get_contents("php://input"), true);

$email = trim($input['email'] ?? '');
$requested_amount = floatval($input['amount'] ?? 0); // optional partial withdraw
$walletaddress = trim($input['walletaddress'] ?? '');
$network = trim($input['network'] ?? '');

// -------------------- Validation --------------------
if (!$email) {
    echo json_encode(["success" => false, "message" => "IB email is required"]);
    exit;
}

if (!$walletaddress) {
    echo json_encode(["success" => false, "message" => "Wallet address is required"]);
    exit;
}

if (!$network) {
    echo json_encode(["success" => false, "message" => "Network is required"]);
    exit;
}

// -------------------- Fetch available profit commission --------------------
$stmt_profit = $conn->prepare("SELECT total_profit_commission, available_profit_commission FROM ib_total_profit_commission WHERE ib_email = ?");
$stmt_profit->bind_param("s", $email);
$stmt_profit->execute();
$result_profit = $stmt_profit->get_result();
$data_profit = $result_profit->fetch_assoc();
$stmt_profit->close();

if (!$data_profit) {
    echo json_encode(["success" => false, "message" => "No profit commission record found for this IB"]);
    exit;
}

$available_profit_commission = floatval($data_profit['available_profit_commission'] ?? 0);

// -------------------- Determine withdraw amount --------------------
$withdraw_amount = $requested_amount > 0 ? $requested_amount : $available_profit_commission;

// -------------------- Validations --------------------
if ($withdraw_amount <= 0) {
    echo json_encode(["success" => false, "message" => "No available profit commission to withdraw"]);
    exit;
}

if ($withdraw_amount > $available_profit_commission) {
    echo json_encode(["success" => false, "message" => "Requested amount exceeds available profit commission"]);
    exit;
}

// -------------------- Generate random reference --------------------
$random = bin2hex(random_bytes(8));

// -------------------- Insert withdraw request --------------------
$insert_stmt = $conn->prepare("INSERT INTO ib_withdraw_request 
    (email, total_commission, walletaddress, network, random, status)
    VALUES (?, ?, ?, ?, ?, 0)");
$insert_stmt->bind_param(
    "sdsss",
    $email,
    $withdraw_amount,
    $walletaddress,
    $network,
    $random
);
$executed = $insert_stmt->execute();
$insert_stmt->close();

if ($executed) {

    echo json_encode([
        "success" => true,
        "message" => "Withdrawal request created successfully",
        "withdraw_amount" => $withdraw_amount,
        "available_profit_commission" => round($available_profit_commission - $withdraw_amount, 4)
    ], JSON_PRETTY_PRINT);
} else {
    echo json_encode(["success" => false, "message" => "Failed to create withdrawal request"]);
}

/* OLD LOGIC (WITHDRAW FROM ib_total_commission)
$amount = floatval($input['amount'] ?? 0);
$withdraw_amount = $amount; // ignored in new logic
*/
?>
