<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

include_once __DIR__ . "/config.php";
include_once __DIR__ . "/connection.php";

$input = json_decode(file_get_contents("php://input"), true);

$email = trim($input['email'] ?? '');
$requested_amount = floatval($input['withdraw_amount'] ?? 0); // optional partial
$withdraw_type = trim($input['withdraw_type'] ?? '');
$company_bank = trim($input['company_bank'] ?? '');
$client_bank = trim($input['client_bank'] ?? '');
$account_holder_name = trim($input['account_holder_name'] ?? '');
$account_number = trim($input['account_number'] ?? '');
$ifsc = trim($input['ifsc'] ?? '');
$type = 'profit_commission'; // always profit commission
$transaction_id = uniqid("TXN");
$random = bin2hex(random_bytes(10));

if (!$email) {
    echo json_encode(["success" => false, "message" => "Email is required"]);
    exit;
}

// -------------------- Fetch profit commission --------------------
$stmt_profit = $conn->prepare("SELECT total_profit_commission, available_profit_commission FROM ib_total_profit_commission WHERE ib_email = ?");
$stmt_profit->bind_param("s", $email);
$stmt_profit->execute();
$result_profit = $stmt_profit->get_result();
$data_profit = $result_profit->fetch_assoc();
$stmt_profit->close();

if (!$data_profit) {
    echo json_encode(["success" => false, "message" => "No profit commission record found for this IB"]);
    exit;
}

$total_profit_commission = floatval($data_profit['total_profit_commission'] ?? 0);
$available_profit_commission = floatval($data_profit['available_profit_commission'] ?? 0);

// -------------------- Determine withdraw amount --------------------
$withdraw_amount = $requested_amount > 0 ? $requested_amount : $available_profit_commission;

// -------------------- Validations --------------------
if ($withdraw_amount <= 0) {
    echo json_encode(["success" => false, "message" => "No available profit commission to withdraw"]);
    exit;
}

if ($withdraw_amount > $available_profit_commission) {
    echo json_encode(["success" => false, "message" => "Requested amount exceeds available profit commission"]);
    exit;
}

// -------------------- Insert withdrawal request --------------------
$insert = $conn->prepare("INSERT INTO ib_withdraw 
    (email, withdraw_amount, withdraw_type, company_bank, client_bank, account_holder_name, 
    account_number, ifsc, type, transaction_id, random, Status) 
    VALUES (?,?,?,?,?,?,?,?,?,?,?,0)");

$insert->bind_param("sdsssssssss", $email, $withdraw_amount, $withdraw_type, $company_bank,
    $client_bank, $account_holder_name, $account_number, $ifsc, $type, $transaction_id, $random);

if ($insert->execute()) {
    // -------------------- Deduct available profit commission --------------------
    $update_stmt = $conn->prepare("
        UPDATE ib_total_profit_commission 
        SET available_profit_commission = available_profit_commission - ?, 
            updated_at = CURRENT_TIMESTAMP 
        WHERE ib_email = ?
    ");
    $update_stmt->bind_param("ds", $withdraw_amount, $email);
    $update_stmt->execute();
    $update_stmt->close();

    echo json_encode([
        "success" => true,
        "message" => "Withdrawal request submitted successfully",
        "transaction_id" => $transaction_id,
        "requested_amount" => $withdraw_amount,
        "available_profit_commission" => round($available_profit_commission - $withdraw_amount, 4)
    ]);
} else {
    echo json_encode([
        "success" => false,
        "message" => "Failed to submit withdrawal request",
        "error" => $insert->error
    ]);
}

/* OLD LOGIC (WITHDRAW FROM ib_total_commission)
$withdraw_amount = floatval($input['withdraw_amount'] ?? 0);
$type = trim($input['type'] ?? 'commission');
*/
?>
